// // // // // // // // // // // // //
//
//	AVLTree.cc
//
//	created April/28/98 by Andreas Warnke
//	modified Mai/3/98 by Andreas Warnke
//



// // // // // // // // // // // // //
//
//	#define
//

//	Set to 1, if you are interested in an debug-output stream.
#define eAVLDebug 0
//	Set to 1, if you want to check parameters for illegal values
#define eAVLParameterCheck 0



// // // // // // // // // // // // //
//
//	include
//

#include "AVLTree.h"
#include "AVLNode.h"
#if eAVLDebug
	//	use libmslcpp.a
	#include <iostream.h>
#endif



// // // // // // // // // // // // //
//
//	ParameterCheck:
//

void AVLParameterCheck ( const char * inText );
void AVLParameterCheck ( const char * inText )
{
	BAlert * sAlert = new BAlert (
		"Unexpected Error",
		inText,
		"?" );
	if ( sAlert != NULL )
		sAlert -> Go ( NULL );
	//	wait forever
	while ( true )
		snooze ( 1000000 );
};



// // // // // // // // // // // // //
//
//	Constructors:
//

AVLTree :: AVLTree ()
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: AVLTree ();\n";
	#endif
	
	//	set default values:
	TheRoot = NULL;
	TheCount = 0;
};



// // // // // // // // // // // // //
//
//	Destructor:
//

AVLTree :: ~AVLTree ()
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: ~AVLTree ();\n";
	#endif
	
	//	remove all nodes:
	MakeEmpty ();
};



// // // // // // // // // // // // //
//
//	MakeEmpty
//

void AVLTree :: MakeEmpty ( bool inDelete )
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: MakeEmpty ( " << inDelete << " );\n";
		cout << "-> removed " << TheCount << " nodes\n";
	#endif
	
	if ( TheRoot != NULL )
		CrashSubTree ( TheRoot, inDelete );
	TheRoot = NULL;
	TheCount = 0;
};



// // // // // // // // // // // // //
//
//	CrashSubTree
//

void AVLTree :: CrashSubTree ( AVLNode * inRoot, bool inDelete )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( inRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/CrashSubTree" );
	#endif

	//	Crash left subtree:
	if ( inRoot -> TheLeft != NULL )
		CrashSubTree ( inRoot -> TheLeft, inDelete );
	inRoot -> TheLeft = NULL;
	
	//	Crash right subtree:
	if ( inRoot -> TheRight != NULL )
		CrashSubTree ( inRoot -> TheRight, inDelete );
	inRoot -> TheRight = NULL;
	
	//	Clear Fields:
	inRoot -> TheTree = NULL;
	inRoot -> TheBalance = 0;
	
	//	delete:
	if ( inDelete )
		delete inRoot;
};



// // // // // // // // // // // // //
//
//	Find
//

AVLNode * AVLTree :: Find ( int inKey )
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: Find ( " << inKey << " );\n";
	#endif
	
	AVLNode * sFinger = TheRoot;
	
	while ( sFinger != NULL )
	{
		//	The result is in the subtree with the root sFinger.
		int sFingerKey = sFinger -> TheKey;
		if ( sFingerKey == inKey )
		{
			//	found.

			//	debug:
			#if ( eAVLDebug )
				cout << "-> found\n";
			#endif

			return sFinger;
		};
		
		if ( inKey > sFingerKey )
			//	result is right:
			sFinger = sFinger -> TheRight;
		else
			//	result is left:
			sFinger = sFinger -> TheLeft;
	};
	
	//	debug:
	#if ( eAVLDebug )
		cout << "-> NULL\n";
	#endif

	//	Nothing to find:
	return NULL;
};



// // // // // // // // // // // // //
//
//	Insert:
//

bool AVLTree :: Insert ( AVLNode * inNode )
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: Insert ( " << inNode -> TheKey << " );\n";
	#endif

	//	Parametercheck:
	if ( inNode == NULL )
		//	error.
		return false;
		
	if ( inNode -> TheTree != NULL )
		//	The node is already in a tree.
		return false;
		
	//	insert:
	InsertInSubtree ( & TheRoot, inNode, this );
	
	//	increase counter:
	TheCount ++;
	
	//	debug:
	#if ( eAVLDebug )
		cout << "-> inserted\n";
		cout_tree ( TheRoot );
	#endif

	//	ready:
	return true;
};



// // // // // // // // // // // // //
//
//	CountNodes:
//

unsigned int AVLTree :: CountNodes ()
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: CountNodes ();\n";
		cout << "-> " << TheCount << "\n";
	#endif

	return TheCount;
};



// // // // // // // // // // // // //
//
//	InsertInSubtree:
//

bool AVLTree :: InsertInSubtree (
	AVLNode ** ioRoot,
	AVLNode * inNode,
	AVLTree * inTree )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/InsertInSubTree" );
	#endif

	//	dereference ioRoot:
	AVLNode * sRoot = * ioRoot;
	
	if ( sRoot == NULL )
	{
		//	Insertation point found.
		//	insert:
		* ioRoot = inNode;
		inNode -> TheTree = inTree;
		inNode -> TheLeft = NULL;
		inNode -> TheRight = NULL;
		inNode -> TheBalance = 0;
		//	ready:
		return true;
	};
	
	//	look for insertation point:
	bool sTreeGrew;
	if ( inNode -> TheKey > sRoot -> TheKey )
	{
		//	insert right:
		sTreeGrew = InsertInSubtree (
			& ( sRoot -> TheRight ),
			inNode,
			inTree );
		if ( ! sTreeGrew )
			//	ready.
			return false;
		
		//	right subtree grew.
		sRoot -> TheBalance ++;
		if ( sRoot -> TheBalance > 1 )
		{
			//	invalid value
			if ( sRoot -> TheRight -> TheBalance >= 0 )
				RotateLeft ( ioRoot );
			else
				RotateRightLeft ( ioRoot );
			return false;
		}
		else if ( sRoot -> TheBalance == 0 )
			return false;
		else
			return true;
	}
	else
	{
		//	insert left:
		sTreeGrew = InsertInSubtree (
			& ( sRoot -> TheLeft ),
			inNode,
			inTree );
		if ( ! sTreeGrew )
			//	ready.
			return false;
		
		//	left subtree grew.
		sRoot -> TheBalance --;
		if ( sRoot -> TheBalance < -1 )
		{
			//	invalid value
			if ( sRoot -> TheLeft -> TheBalance <= 0 )
				RotateRight ( ioRoot );
			else
				RotateLeftRight ( ioRoot );
			return false;
		}
		else if ( sRoot -> TheBalance == 0 )
			return false;
		else
			return true;
	};
};



// // // // // // // // // // // // //
//
//	RotateLeft:
//

void AVLTree :: RotateLeft ( AVLNode ** ioRoot )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateLeft/1" );
		else if ( *ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateLeft/2" );
		else if ( (*ioRoot) -> TheRight == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateLeft/3" );
	#endif

	//	Copy Data to stack:
	AVLNode * sRoot = * ioRoot;
	AVLNode * sRight = sRoot -> TheRight;
	int sRootBal = sRoot -> TheBalance;
	int sRightBal = sRight -> TheBalance;
	
	//	Rotation:
	sRoot -> TheRight = sRight -> TheLeft;
	sRight -> TheLeft = sRoot;
	* ioRoot = sRight;
	
	//	Balance Correction:
	int sNewRootBal;
	if ( sRightBal > 0 )
		sNewRootBal = sRootBal - 1 - sRightBal;
	else
		sNewRootBal = sRootBal - 1;
	sRoot -> TheBalance = sNewRootBal;
	if ( sNewRootBal >= 0 )
		sRight -> TheBalance = sRightBal - 1;
	else
		sRight -> TheBalance = sRightBal - 1 + sNewRootBal;
};



// // // // // // // // // // // // //
//
//	RotateRight:
//

void AVLTree :: RotateRight ( AVLNode ** ioRoot )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateRight/1" );
		else if ( *ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateRight/2" );
		else if ( (*ioRoot) -> TheLeft == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateRight/3" );
	#endif

	//	Copy Data to stack:
	AVLNode * sRoot = * ioRoot;
	AVLNode * sLeft = sRoot -> TheLeft;
	int sRootBal = sRoot -> TheBalance;
	int sLeftBal = sLeft -> TheBalance;
	
	//	Rotation:
	sRoot -> TheLeft = sLeft -> TheRight;
	sLeft -> TheRight = sRoot;
	* ioRoot = sLeft;
	
	//	Balance Correction:
	int sNewRootBal;
	if ( sLeftBal >= 0 )
		sNewRootBal = sRootBal + 1;
	else
		sNewRootBal = sRootBal + 1 - sLeftBal;
	sRoot -> TheBalance = sNewRootBal;
	if ( sNewRootBal > 0 )
		sLeft -> TheBalance = sLeftBal + 1 + sNewRootBal;
	else
		sLeft -> TheBalance = sLeftBal + 1;
};



// // // // // // // // // // // // //
//
//	RotateLeftRight:
//

void AVLTree :: RotateLeftRight ( AVLNode ** ioRoot )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateLeftRight/1" );
		else if ( *ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateLeftRight/2" );
	#endif

	RotateLeft ( & ( ( * ioRoot ) -> TheLeft ) );
	RotateRight ( ioRoot );
};



// // // // // // // // // // // // //
//
//	RotateRightLeft:
//

void AVLTree :: RotateRightLeft ( AVLNode ** ioRoot )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateRightLeft/1" );
		else if ( *ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateRightLeft/2" );
	#endif

	RotateRight ( & ( ( * ioRoot ) -> TheRight ) );
	RotateLeft ( ioRoot );
};



// // // // // // // // // // // // //
//
//	Remove ( inNode ):
//

bool AVLTree :: Remove ( AVLNode * inNode )
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: Remove ( " << inNode -> TheKey << " );\n";
	#endif

	//	parametercheck:
	if ( inNode == NULL )
		return false;
	
	//	remove item:	
	AVLNode * sFound;
	RemoveNodeFromSubTree (
		& TheRoot,
		inNode -> TheKey,
		inNode,
		false,
		& sFound );
	
	if ( sFound != NULL )
	{
		//	item removed.

		//	debug:
		#if ( eAVLDebug )
			cout << "-> removed\n";
			cout_tree ( TheRoot );
		#endif

		TheCount --;
		return true;
	}
	else
		//	error.
		return false;
};



// // // // // // // // // // // // //
//
//	Remove ( inKey )
//

AVLNode * AVLTree :: Remove ( int inKey, bool inAll )
{
	//	debug:
	#if ( eAVLDebug )
		cout << "AVLTree :: Remove ( " << inKey << ", " << inAll << " );\n";
	#endif

	//	Remove all inKEy-Nodes:
	AVLNode * sResult = NULL;
	AVLNode * sRemoved;
	do
	{
		//	remove one inKey-Node:
		RemoveNodeFromSubTree (
			& TheRoot,
			inKey,
			NULL,
			false,
			& sRemoved );
		if ( sRemoved != NULL )
		{
			sResult = sRemoved;
			TheCount --;
		};
	}
	while ( inAll && ( sRemoved != NULL ) );
	
	//	debug:
	#if ( eAVLDebug  )
		if ( sResult != NULL )
			cout << "-> removed\n";
		cout_tree ( TheRoot );
	#endif

	//	ready:
	return sResult;
};



// // // // // // // // // // // // //
//
//	RemoveNodeFromSubTree
//	returns true <=> Tree shrunk
//

bool AVLTree :: RemoveNodeFromSubTree (
	AVLNode ** ioRoot,
	int inKey,
	AVLNode * inNode,
	bool inIgnoreKeyAndRemoveLeft,
	AVLNode ** outRemoved )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RemoveNodeFromSubTree/1" );
		else if ( outRemoved == NULL )
			AVLParameterCheck ( "AVLTree.cc/RemoveNodeFromSubTree/2" );
		else if ( inIgnoreKeyAndRemoveLeft && ( inNode != NULL ) )
			AVLParameterCheck ( "AVLTree.cc/RemoveNodeFromSubTree/3" );
	#endif

	//	parametercheck:
	AVLNode * sRoot = * ioRoot;
	if ( sRoot == NULL )
	{
		//	no node to delete:
		* outRemoved = NULL;
		return false;
	};
	
	//	save, if one of the subtrees shrunk:
	bool sLeftShrunk = false;
	bool sRightShrunk = false;
	
	if ( ( ( ! inIgnoreKeyAndRemoveLeft ) && ( inKey == sRoot -> TheKey ) )
		|| ( inIgnoreKeyAndRemoveLeft && ( sRoot -> TheLeft == NULL ) ) )
		//	sRoot has the same key as the node that should be removed.
		if ( ( inNode == NULL ) || ( inNode == sRoot ) )
		{
			//	sRoot is the node that is to be removed.
			* outRemoved = sRoot;
			return RemoveRoot ( ioRoot );
		}
		else
		{
			//	sRoot has the correct key but is not the searched node.
			//	look for inNode in left subtree:
			sLeftShrunk = RemoveNodeFromSubTree (
				& ( sRoot -> TheLeft ),
				inKey,
				inNode,
				inIgnoreKeyAndRemoveLeft,
				outRemoved );
			if ( * outRemoved == NULL )
				//	node not found in left subtree.
				sRightShrunk = RemoveNodeFromSubTree (
					& ( sRoot -> TheRight ),
					inKey,
					inNode,
					inIgnoreKeyAndRemoveLeft,
					outRemoved );
		}
	else
		//	sRoot hasn't the same key as the node to be removed.
		if ( sRoot -> TheKey > inKey )
			//	the node is in the left subtree:
			sLeftShrunk = RemoveNodeFromSubTree (
				& ( sRoot -> TheLeft ),
				inKey,
				inNode,
				inIgnoreKeyAndRemoveLeft,
				outRemoved );
		else
			//	the node is in the right subtree.
			sRightShrunk = RemoveNodeFromSubTree (
				& ( sRoot -> TheRight ),
				inKey,
				inNode,
				inIgnoreKeyAndRemoveLeft,
				outRemoved );
	
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( sLeftShrunk && sRightShrunk )
			AVLParameterCheck ( "AVLTree.cc/RemoveNodeFromSubTree/4" );
	#endif

	//	check the balance factor:
	if ( sLeftShrunk )
	{
		sRoot -> TheBalance ++;
		return RotateAfterRemove ( ioRoot );
	};
	if ( sRightShrunk )
	{
		sRoot -> TheBalance --;
		return RotateAfterRemove ( ioRoot );
	};
	
	//	no subtrees are shrunk:
	return false;
};



// // // // // // // // // // // // //
//
//	RemoveRoot
//	returns true <=> Tree shrunk
//

bool AVLTree :: RemoveRoot ( AVLNode ** ioRoot )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RemoveRoot/1" );
		else if ( *ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RemoveRoot/2" );
	#endif

	AVLNode * sRoot = * ioRoot;
	
	if ( sRoot -> TheLeft == NULL )
	{
		//	no left subtree:
		* ioRoot = sRoot -> TheRight;
		sRoot -> TheRight = NULL;
		sRoot -> TheBalance = 0;
		sRoot -> TheTree = NULL;
		return true;
	};
	
	if ( sRoot -> TheRight == NULL )
	{
		//	no right subtree:
		* ioRoot = sRoot -> TheLeft;
		sRoot -> TheLeft = NULL;
		sRoot -> TheBalance = 0;
		sRoot -> TheTree = NULL;
		return true;
	};
	
	//	both subtrees exist.
	//	remove symmetric successor:
	AVLNode * sSymSuccessor;
	bool sRightShrunk = RemoveNodeFromSubTree (
		& ( sRoot -> TheRight ),
		0,
		NULL,
		true,
		& sSymSuccessor );
	
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( sSymSuccessor == NULL )
			AVLParameterCheck ( "AVLTree.cc/RemoveRoot/3" );
	#endif

	//	exchange sRoot and sSymSuccessor:
	sSymSuccessor -> TheLeft = sRoot -> TheLeft;
	sSymSuccessor -> TheRight = sRoot -> TheRight;
	sSymSuccessor -> TheBalance = sRoot -> TheBalance;
	sSymSuccessor -> TheTree = sRoot -> TheTree;
	* ioRoot = sSymSuccessor;
	sRoot -> TheLeft = NULL;
	sRoot -> TheRight = NULL;
	sRoot -> TheBalance = 0;
	sRoot -> TheTree = NULL;
	
	if ( sRightShrunk )
	{
		//	the right subtree is shrunk.
		sSymSuccessor -> TheBalance --;
		return RotateAfterRemove ( ioRoot );
	};
	
	//	no tree is shrunk.
	return false;
};



// // // // // // // // // // // // //
//
//	RotateAferRemove
//	returns true <=> Tree shrunk
//

bool AVLTree :: RotateAfterRemove ( AVLNode ** ioRoot )
{
	//	ParameterCheck:
	#if eAVLParameterCheck
		if ( ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateAfterRemove/1" );
		else if ( *ioRoot == NULL )
			AVLParameterCheck ( "AVLTree.cc/RotateAfterRemove/2" );
	#endif

	//	Load data to stack:
	AVLNode * sRoot = * ioRoot;
	int sBal = sRoot -> TheBalance;
	
	if ( sBal == 0 )
		//	the balance was 1 or -1 before the remove
		//	means: the ex-longer subtree shrunk
		//	means: the whole tree shrunk
		return true;
		
	if ( ( sBal == 1 ) || ( sBal == -1 ) )
		//	the balance was 0 before the remove
		//	means: the tree didn't not shrink
		return false;
		
	if ( sBal > 1 )
	{
		//	the right subtree is much longer than the left:
		int sRightBal = sRoot -> TheRight -> TheBalance;
		if ( sRightBal == 0 )
		{
			//	Ottmann, Widmayer - p. 309 - case 1.3.1
			RotateLeft ( ioRoot );
			return false;
		};
		if ( sRightBal > 0 )
		{
			//	Ottmann, Widmayer - p. 309 - case 1.3.2
			RotateLeft ( ioRoot );
			return true;
		};
		// if ( sRightBal < 0 )
		{
			//	Ottmann, Widmayer - p. 310 - case 1.3.3
			RotateRightLeft ( ioRoot );
			return true;
		};
	}
	else
	{
		//	the left subtree is much longer than the right:
		int sLeftBal = sRoot -> TheLeft -> TheBalance;
		if ( sLeftBal == 0 )
		{
			//	Ottmann, Widmayer - p. 309 - case 2.3.1
			RotateRight ( ioRoot );
			return false;
		};
		if ( sLeftBal < 0 )
		{
			//	Ottmann, Widmayer - p. 309 - case 2.3.2
			RotateRight ( ioRoot );
			return true;
		};
		// if ( sLeftBal > 0 )
		{
			//	Ottmann, Widmayer - p. 310 - case 2.3.3
			RotateLeftRight ( ioRoot );
			return true;
		};
	};	
};



// // // // // // // // // // // // //
//
//	debug:
//	cout_tree
//

#if eAVLDebug
void cout_tab ( int inLevel );
void cout_tab ( int inLevel )
{ if ( inLevel <= 0 ) return; cout << "      "; cout_tab ( inLevel - 1 ); };
#endif

void AVLTree :: cout_tree ( AVLNode * inRoot, int inLevel )
{
#if eAVLDebug
	if ( inRoot != NULL )
	{
		cout_tree ( inRoot -> TheRight, inLevel + 1 );
		cout_tab ( inLevel );
		cout << inRoot -> TheKey << " " << inRoot -> TheBalance << "\n";
		cout_tree ( inRoot -> TheLeft, inLevel + 1 );
	};
#endif
};



//
//	The End
//
// // // // // // // // // // // // //